<?php
/**
 * Tooltastic Sales Popup – Frontend Script Injection (v1.1.0)
 *
 * Injects the widget script into the frontend.
 * Supports multi-campaign page matching + WooCommerce integration.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Tooltastic_SP_Frontend {

    public static function init() {
        add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_widget']);

        // WooCommerce integration
        if (get_option('tooltastic_sp_woo_integration', '0') === '1') {
            add_action('woocommerce_thankyou', [__CLASS__, 'send_woo_event']);
        }
    }

    /**
     * Enqueue the widget script on the frontend.
     */
    public static function enqueue_widget() {
        if (get_option('tooltastic_sp_enabled', '1') !== '1') {
            return;
        }

        $api_key = get_option('tooltastic_sp_api_key', '');
        if (empty($api_key)) {
            return;
        }

        if (self::is_excluded_page()) {
            return;
        }

        if (is_admin() || wp_doing_cron()) {
            return;
        }

        // Determine the campaign_id for the current page
        $campaign_id = self::get_matching_campaign_id();

        wp_enqueue_script(
            'tooltastic-sales-popup',
            TOOLTASTIC_SP_WIDGET_URL,
            [],
            null,
            true
        );

        add_filter('script_loader_tag', function($tag, $handle) use ($api_key, $campaign_id) {
            if ($handle === 'tooltastic-sales-popup') {
                $attrs = 'data-key="' . esc_attr($api_key) . '"';
                if (!empty($campaign_id)) {
                    $attrs .= ' data-campaign="' . esc_attr($campaign_id) . '"';
                }
                $tag = str_replace(' src=', ' ' . $attrs . ' src=', $tag);
            }
            return $tag;
        }, 10, 2);
    }

    /**
     * Get the matching campaign ID for the current page.
     *
     * Evaluates campaigns in order — first match wins:
     *   - "include" → matches if current path is in the page list
     *   - "exclude" → matches if current path is NOT in the page list
     *   - "all"     → always matches (catch-all / default)
     */
    public static function get_matching_campaign_id() {
        $campaigns = get_option('tooltastic_sp_campaigns', []);

        if (!is_array($campaigns) || empty($campaigns)) {
            // Backward compat: fall back to old single campaign_id
            return get_option('tooltastic_sp_campaign_id', '');
        }

        $current_path = wp_parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);

        foreach ($campaigns as $campaign) {
            if (empty($campaign['campaign_id'])) {
                continue;
            }

            $mode = $campaign['mode'] ?? 'all';

            if ($mode === 'all') {
                return $campaign['campaign_id'];
            }

            $pages = array_filter(array_map('trim', explode("\n", $campaign['pages'] ?? '')));

            if (empty($pages)) {
                // No pages defined for include/exclude → skip this campaign
                continue;
            }

            $matches_page = self::path_matches_patterns($current_path, $pages);

            if ($mode === 'include' && $matches_page) {
                return $campaign['campaign_id'];
            }

            if ($mode === 'exclude' && !$matches_page) {
                return $campaign['campaign_id'];
            }
        }

        // No campaign matched
        return '';
    }

    /**
     * Check if a path matches any of the given patterns.
     */
    private static function path_matches_patterns($path, $patterns) {
        $path = rtrim($path, '/');

        foreach ($patterns as $pattern) {
            if (empty($pattern)) {
                continue;
            }

            if (strpos($pattern, '*') !== false) {
                $regex = '#^' . str_replace('\*', '.*', preg_quote($pattern, '#')) . '$#';
                if (preg_match($regex, $path)) {
                    return true;
                }
                // Also try with trailing slash removed from pattern
                $pattern_trimmed = rtrim($pattern, '/');
                if ($pattern_trimmed !== $pattern) {
                    $regex2 = '#^' . str_replace('\*', '.*', preg_quote($pattern_trimmed, '#')) . '$#';
                    if (preg_match($regex2, $path)) {
                        return true;
                    }
                }
            } else {
                $pattern_clean = rtrim($pattern, '/');
                if ($path === $pattern_clean) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Check if the current page is globally excluded.
     */
    private static function is_excluded_page() {
        $exclude_pages = get_option('tooltastic_sp_exclude_pages', '');
        if (empty($exclude_pages)) {
            return false;
        }

        $current_path = wp_parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
        $excludes = array_filter(array_map('trim', explode("\n", $exclude_pages)));

        return self::path_matches_patterns($current_path, $excludes);
    }

    /**
     * WooCommerce: Send purchase event to Tooltastic API.
     */
    public static function send_woo_event($order_id) {
        $api_key = get_option('tooltastic_sp_api_key', '');
        if (empty($api_key)) {
            return;
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        $status = $order->get_status();
        if (in_array($status, ['failed', 'cancelled', 'refunded'], true)) {
            return;
        }

        $items = $order->get_items();
        $product_name = '';
        $product_image = '';
        $product_url = '';
        $product_price = '';

        foreach ($items as $item) {
            $product = $item->get_product();
            if ($product) {
                $product_name = $product->get_name();
                $product_url = $product->get_permalink();
                $product_price = wc_price($product->get_price());
                $image_id = $product->get_image_id();
                if ($image_id) {
                    $product_image = wp_get_attachment_image_url($image_id, 'thumbnail');
                }
                break;
            }
        }

        $first_name = $order->get_billing_first_name();
        $city = $order->get_billing_city();

        $event_data = [
            'key'  => $api_key,
            'type' => 'sale',
            'data' => [
                'name'          => $first_name ?: 'Kunde',
                'city'          => $city ?: '',
                'product_name'  => $product_name,
                'product_image' => $product_image,
                'product_url'   => $product_url,
                'product_price' => wp_strip_all_tags($product_price),
            ],
        ];

        wp_remote_post(
            TOOLTASTIC_SP_API_URL . '/event',
            [
                'timeout'   => 5,
                'blocking'  => false,
                'sslverify' => true,
                'headers'   => ['Content-Type' => 'application/json'],
                'body'      => wp_json_encode($event_data),
            ]
        );
    }
}
