<?php
/**
 * Tooltastic Sales Popup – Admin Settings Page (v1.1.0)
 *
 * Premium tabbed interface with multi-campaign support.
 * Settings under: Einstellungen → Tooltastic Sales Popup
 */

if (!defined('ABSPATH')) {
    exit;
}

class Tooltastic_SP_Settings {

    public static function init() {
        add_action('admin_menu', [__CLASS__, 'add_menu']);
        add_action('admin_init', [__CLASS__, 'register_settings']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_admin_assets']);

        // AJAX: Test API key
        add_action('wp_ajax_tooltastic_sp_test_key', [__CLASS__, 'ajax_test_key']);
    }

    /**
     * Add settings page.
     */
    public static function add_menu() {
        add_options_page(
            'Tooltastic Sales Popup',
            'Tooltastic Sales Popup',
            'manage_options',
            'tooltastic-sales-popup',
            [__CLASS__, 'render_page']
        );
    }

    /**
     * Register settings.
     */
    public static function register_settings() {
        register_setting('tooltastic_sp', 'tooltastic_sp_api_key', [
            'sanitize_callback' => 'sanitize_text_field',
        ]);
        register_setting('tooltastic_sp', 'tooltastic_sp_campaigns', [
            'sanitize_callback' => [__CLASS__, 'sanitize_campaigns'],
        ]);
        register_setting('tooltastic_sp', 'tooltastic_sp_enabled', [
            'sanitize_callback' => function($val) { return $val ? '1' : '0'; },
        ]);
        register_setting('tooltastic_sp', 'tooltastic_sp_exclude_pages', [
            'sanitize_callback' => 'sanitize_textarea_field',
        ]);
        register_setting('tooltastic_sp', 'tooltastic_sp_woo_integration', [
            'sanitize_callback' => function($val) { return $val ? '1' : '0'; },
        ]);

        // Keep old setting registered for backward compat
        register_setting('tooltastic_sp', 'tooltastic_sp_campaign_id', [
            'sanitize_callback' => 'sanitize_text_field',
        ]);
    }

    /**
     * Sanitize the campaigns array input.
     */
    public static function sanitize_campaigns($input) {
        if (!is_array($input)) {
            return [];
        }

        $clean = [];
        foreach ($input as $campaign) {
            if (!is_array($campaign)) {
                continue;
            }

            $mode = $campaign['mode'] ?? 'all';
            if (!in_array($mode, ['all', 'include', 'exclude'], true)) {
                $mode = 'all';
            }

            $c = [
                'label'       => sanitize_text_field($campaign['label'] ?? ''),
                'campaign_id' => sanitize_text_field($campaign['campaign_id'] ?? ''),
                'mode'        => $mode,
                'pages'       => sanitize_textarea_field($campaign['pages'] ?? ''),
            ];

            // Only save entries with a campaign_id
            if (!empty($c['campaign_id'])) {
                $clean[] = $c;
            }
        }

        return $clean;
    }

    /**
     * Enqueue admin CSS + JS.
     */
    public static function enqueue_admin_assets($hook) {
        if ($hook !== 'settings_page_tooltastic-sales-popup') {
            return;
        }

        wp_enqueue_style(
            'tooltastic-sp-admin',
            TOOLTASTIC_SP_URL . 'admin/admin.css',
            [],
            TOOLTASTIC_SP_VERSION
        );
    }

    /**
     * AJAX: Test API key connection.
     */
    public static function ajax_test_key() {
        check_ajax_referer('tooltastic_sp_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => esc_html__('Keine Berechtigung.', 'tooltastic-sales-popup')]);
        }

        $key = sanitize_text_field(wp_unslash($_POST['api_key'] ?? ''));
        if (empty($key)) {
            wp_send_json_error(['message' => esc_html__('Bitte API Key eingeben.', 'tooltastic-sales-popup')]);
        }

        $response = wp_remote_get(
            TOOLTASTIC_SP_API_URL . '/sales-popup?key=' . urlencode($key),
            ['timeout' => 10, 'sslverify' => true]
        );

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => 'Verbindungsfehler: ' . esc_html($response->get_error_message())]);
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code === 200 && !empty($body['success'])) {
            $mode = sanitize_text_field($body['mode'] ?? 'normal');
            wp_send_json_success([
                'message' => sprintf(
                    /* translators: %s: widget mode */
                    esc_html__('Verbindung erfolgreich! Modus: %s', 'tooltastic-sales-popup'),
                    $mode
                ),
                'mode' => $mode,
            ]);
        } else {
            $error = sanitize_text_field($body['error'] ?? 'unknown');
            $messages = [
                'invalid_key'   => __('Ungültiger API Key.', 'tooltastic-sales-popup'),
                'key_disabled'  => __('API Key ist deaktiviert.', 'tooltastic-sales-popup'),
                'tool_disabled' => __('Das Tool ist derzeit deaktiviert.', 'tooltastic-sales-popup'),
                'maintenance'   => __('Das Tool befindet sich im Wartungsmodus.', 'tooltastic-sales-popup'),
            ];
            $msg = isset($messages[$error]) ? $messages[$error] : ('Fehler: ' . $error);
            wp_send_json_error(['message' => esc_html($msg)]);
        }
    }

    /**
     * Render settings page.
     */
    public static function render_page() {
        $api_key         = get_option('tooltastic_sp_api_key', '');
        $campaigns       = get_option('tooltastic_sp_campaigns', []);
        $enabled         = get_option('tooltastic_sp_enabled', '1');
        $exclude_pages   = get_option('tooltastic_sp_exclude_pages', '');
        $woo_integration = get_option('tooltastic_sp_woo_integration', '0');
        $woo_active      = class_exists('WooCommerce');

        if (!is_array($campaigns)) {
            $campaigns = [];
        }
        ?>
        <div class="wrap tooltastic-sp-wrap">

            <!-- ── Header ──────────────────────────────────── -->
            <div class="tsp-header">
                <div class="tsp-header-left">
                    <div class="tsp-logo">
                        <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M6 8a6 6 0 0 1 12 0c0 7 3 9 3 9H3s3-2 3-9"/>
                            <path d="M10.3 21a1.94 1.94 0 0 0 3.4 0"/>
                        </svg>
                    </div>
                    <div class="tsp-header-text">
                        <h1><?php echo esc_html__('Tooltastic Sales Popup', 'tooltastic-sales-popup'); ?></h1>
                        <p><?php echo esc_html__('Social-Proof Benachrichtigungen für deine Website', 'tooltastic-sales-popup'); ?></p>
                    </div>
                </div>
                <div class="tsp-header-right">
                    <span class="tsp-version">v<?php echo esc_html(TOOLTASTIC_SP_VERSION); ?></span>
                    <span class="tsp-status <?php echo !empty($api_key) ? 'tsp-status--ok' : 'tsp-status--empty'; ?>" id="tsp-status">
                        <?php if (!empty($api_key)): ?>
                            <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><polyline points="20 6 9 17 4 12"/></svg>
                            <?php echo esc_html__('Verbunden', 'tooltastic-sales-popup'); ?>
                        <?php else: ?>
                            <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>
                            <?php echo esc_html__('Nicht verbunden', 'tooltastic-sales-popup'); ?>
                        <?php endif; ?>
                    </span>
                </div>
            </div>

            <?php
            // Show only one success notice (avoid WP duplicating for each registered setting)
            if (isset($_GET['settings-updated']) && $_GET['settings-updated'] === 'true') {
                echo '<div class="tsp-message tsp-message--success" style="margin:12px 0">';
                echo '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>';
                echo '<span>' . esc_html__('Die Einstellungen wurden gespeichert.', 'tooltastic-sales-popup') . '</span>';
                echo '</div>';
            } else {
                settings_errors('tooltastic_sp');
            }
            ?>


            <!-- ── Tabs ────────────────────────────────────── -->
            <nav class="tsp-tabs" role="tablist">
                <button type="button" class="tsp-tab tsp-tab--active" data-tab="connection" role="tab" aria-selected="true">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M15 3h4a2 2 0 0 1 2 2v14a2 2 0 0 1-2 2h-4"/><polyline points="10 17 15 12 10 7"/><line x1="15" y1="12" x2="3" y2="12"/></svg>
                    <?php echo esc_html__('Verbindung', 'tooltastic-sales-popup'); ?>
                </button>
                <button type="button" class="tsp-tab" data-tab="campaigns" role="tab" aria-selected="false">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="7" height="7" rx="1"/><rect x="14" y="3" width="7" height="7" rx="1"/><rect x="3" y="14" width="7" height="7" rx="1"/><rect x="14" y="14" width="7" height="7" rx="1"/></svg>
                    <?php echo esc_html__('Kampagnen', 'tooltastic-sales-popup'); ?>
                    <?php if (!empty($campaigns)): ?>
                        <span class="tsp-tab-badge"><?php echo count($campaigns); ?></span>
                    <?php endif; ?>
                </button>
                <button type="button" class="tsp-tab" data-tab="advanced" role="tab" aria-selected="false">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="3"/><path d="M12 1v2M12 21v2M4.22 4.22l1.42 1.42M18.36 18.36l1.42 1.42M1 12h2M21 12h2M4.22 19.78l1.42-1.42M18.36 5.64l1.42-1.42"/></svg>
                    <?php echo esc_html__('Erweitert', 'tooltastic-sales-popup'); ?>
                </button>
            </nav>

            <!-- ── Form ────────────────────────────────────── -->
            <form method="post" action="options.php" id="tsp-form">
                <?php settings_fields('tooltastic_sp'); ?>

                <!-- Tab: Connection ──────────────────────── -->
                <div class="tsp-panel tsp-panel--active" data-panel="connection">
                    <div class="tsp-card">
                        <div class="tsp-card-header">
                            <h2><?php echo esc_html__('API-Verbindung', 'tooltastic-sales-popup'); ?></h2>
                            <p><?php echo esc_html__('Verbinde dein Tooltastic-Konto mit deiner WordPress-Seite.', 'tooltastic-sales-popup'); ?></p>
                        </div>

                        <div class="tsp-field">
                            <label class="tsp-label" for="tooltastic_sp_api_key"><?php echo esc_html__('API Key', 'tooltastic-sales-popup'); ?></label>
                            <div class="tsp-input-group">
                                <input type="text"
                                       id="tooltastic_sp_api_key"
                                       name="tooltastic_sp_api_key"
                                       value="<?php echo esc_attr($api_key); ?>"
                                       class="tsp-input tsp-input--code"
                                       placeholder="tt_xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"
                                       autocomplete="off">
                                <button type="button" id="tsp-test-key" class="tsp-btn tsp-btn--secondary">
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="23 4 23 10 17 10"/><path d="M20.49 15a9 9 0 1 1-2.12-9.36L23 10"/></svg>
                                    <?php echo esc_html__('Testen', 'tooltastic-sales-popup'); ?>
                                </button>
                            </div>
                            <div id="tsp-test-result" class="tsp-message" style="display:none"></div>
                            <p class="tsp-hint">
                                <?php
                                printf(
                                    /* translators: %s: link to dashboard */
                                    esc_html__('Deinen API Key findest du im %s.', 'tooltastic-sales-popup'),
                                    '<a href="https://tooltastic.eu/de/dashboard/api-zugangsdaten/" target="_blank" rel="noopener">' .
                                    esc_html__('Tooltastic Dashboard', 'tooltastic-sales-popup') .
                                    ' <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="vertical-align:-1px"><path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"/><polyline points="15 3 21 3 21 9"/><line x1="10" y1="14" x2="21" y2="3"/></svg></a>'
                                );
                                ?>
                            </p>
                        </div>
                    </div>

                    <!-- Quick start -->
                    <div class="tsp-card tsp-card--info">
                        <h3>
                            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><path d="M9.09 9a3 3 0 0 1 5.83 1c0 2-3 3-3 3"/><line x1="12" y1="17" x2="12.01" y2="17"/></svg>
                            <?php echo esc_html__('Schnellstart', 'tooltastic-sales-popup'); ?>
                        </h3>
                        <ol>
                            <li><?php
                                printf(
                                    esc_html__('Erstelle einen API Key im %s.', 'tooltastic-sales-popup'),
                                    '<a href="https://tooltastic.eu/de/dashboard/api-zugangsdaten/" target="_blank" rel="noopener">' . esc_html__('Tooltastic Dashboard', 'tooltastic-sales-popup') . '</a>'
                                );
                            ?></li>
                            <li><?php
                                printf(
                                    esc_html__('Erstelle eine Kampagne im %s.', 'tooltastic-sales-popup'),
                                    '<a href="https://tooltastic.eu/de/dashboard/sales-popup/" target="_blank" rel="noopener">' . esc_html__('Kampagnen-Manager', 'tooltastic-sales-popup') . '</a>'
                                );
                            ?></li>
                            <li><?php echo esc_html__('Füge den API Key oben ein, richte deine Kampagnen ein — fertig!', 'tooltastic-sales-popup'); ?></li>
                        </ol>
                    </div>
                </div>

                <!-- Tab: Campaigns ───────────────────────── -->
                <div class="tsp-panel" data-panel="campaigns">
                    <div class="tsp-card">
                        <div class="tsp-card-header">
                            <h2><?php echo esc_html__('Kampagnen-Zuordnung', 'tooltastic-sales-popup'); ?></h2>
                            <p><?php echo esc_html__('Weise verschiedene Kampagnen bestimmten Seiten zu. Die erste passende Kampagne wird verwendet.', 'tooltastic-sales-popup'); ?></p>
                        </div>

                        <div id="tsp-campaigns">
                            <?php if (empty($campaigns)): ?>
                                <!-- Empty state rendered by JS on load -->
                            <?php else: ?>
                                <?php foreach ($campaigns as $i => $campaign): ?>
                                    <?php self::render_campaign_card($i, $campaign); ?>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>

                        <div id="tsp-empty-state" class="tsp-empty" <?php echo !empty($campaigns) ? 'style="display:none"' : ''; ?>>
                            <div class="tsp-empty-icon">
                                <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5"><rect x="3" y="3" width="7" height="7" rx="1"/><rect x="14" y="3" width="7" height="7" rx="1"/><rect x="3" y="14" width="7" height="7" rx="1"/><rect x="14" y="14" width="7" height="7" rx="1"/></svg>
                            </div>
                            <h3><?php echo esc_html__('Noch keine Kampagnen eingerichtet', 'tooltastic-sales-popup'); ?></h3>
                            <p><?php echo esc_html__('Erstelle deine erste Kampagnen-Zuordnung, um Social-Proof Popups auf deiner Website anzuzeigen.', 'tooltastic-sales-popup'); ?></p>
                        </div>

                        <button type="button" id="tsp-add-campaign" class="tsp-btn tsp-btn--add">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="12" y1="5" x2="12" y2="19"/><line x1="5" y1="12" x2="19" y2="12"/></svg>
                            <?php echo esc_html__('Kampagne hinzufügen', 'tooltastic-sales-popup'); ?>
                        </button>
                    </div>

                    <div class="tsp-card tsp-card--tip">
                        <h3>
                            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 2L2 7l10 5 10-5-10-5z"/><path d="M2 17l10 5 10-5"/><path d="M2 12l10 5 10-5"/></svg>
                            <?php echo esc_html__('Tipp: Mehrere Kampagnen', 'tooltastic-sales-popup'); ?>
                        </h3>
                        <p><?php echo esc_html__('Zeige unterschiedliche Popups auf verschiedenen Bereichen deiner Website:', 'tooltastic-sales-popup'); ?></p>
                        <ul>
                            <li><?php echo esc_html__('Shop-Seiten: Kauf-Benachrichtigungen', 'tooltastic-sales-popup'); ?></li>
                            <li><?php echo esc_html__('Blog: Anmeldungen & Reviews', 'tooltastic-sales-popup'); ?></li>
                            <li><?php echo esc_html__('Landing Pages: Knappheit & Social Proof', 'tooltastic-sales-popup'); ?></li>
                        </ul>
                        <p class="tsp-hint"><?php echo esc_html__('Die Kampagnen werden von oben nach unten geprüft — die erste passende Zuordnung gewinnt.', 'tooltastic-sales-popup'); ?></p>
                    </div>
                </div>

                <!-- Tab: Advanced ────────────────────────── -->
                <div class="tsp-panel" data-panel="advanced">
                    <div class="tsp-card">
                        <div class="tsp-card-header">
                            <h2><?php echo esc_html__('Erweiterte Einstellungen', 'tooltastic-sales-popup'); ?></h2>
                        </div>

                        <div class="tsp-field">
                            <label class="tsp-toggle-label">
                                <span class="tsp-toggle">
                                    <input type="hidden" name="tooltastic_sp_enabled" value="0">
                                    <input type="checkbox"
                                           name="tooltastic_sp_enabled"
                                           value="1"
                                           <?php checked($enabled, '1'); ?>>
                                    <span class="tsp-toggle-slider"></span>
                                </span>
                                <span>
                                    <strong><?php echo esc_html__('Widget aktiv', 'tooltastic-sales-popup'); ?></strong>
                                    <small><?php echo esc_html__('Sales Popup auf dieser Website anzeigen', 'tooltastic-sales-popup'); ?></small>
                                </span>
                            </label>
                        </div>

                        <hr class="tsp-divider">

                        <div class="tsp-field">
                            <label class="tsp-label" for="tooltastic_sp_exclude_pages"><?php echo esc_html__('Seiten ausschließen', 'tooltastic-sales-popup'); ?></label>
                            <textarea id="tooltastic_sp_exclude_pages"
                                      name="tooltastic_sp_exclude_pages"
                                      rows="4"
                                      class="tsp-textarea tsp-input--code"
                                      placeholder="/warenkorb/&#10;/checkout/&#10;/kasse/&#10;/mein-konto/"><?php echo esc_textarea($exclude_pages); ?></textarea>
                            <p class="tsp-hint">
                                <?php echo esc_html__('Ein URL-Pfad pro Zeile. Das Widget wird auf diesen Seiten nicht geladen.', 'tooltastic-sales-popup'); ?>
                                <?php echo esc_html__('Unterstützt Wildcards:', 'tooltastic-sales-popup'); ?> <code>/shop/*</code>
                            </p>
                        </div>

                        <?php if ($woo_active): ?>
                        <hr class="tsp-divider">

                        <div class="tsp-field">
                            <label class="tsp-toggle-label">
                                <span class="tsp-toggle">
                                    <input type="hidden" name="tooltastic_sp_woo_integration" value="0">
                                    <input type="checkbox"
                                           name="tooltastic_sp_woo_integration"
                                           value="1"
                                           <?php checked($woo_integration, '1'); ?>>
                                    <span class="tsp-toggle-slider"></span>
                                </span>
                                <span>
                                    <strong>
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#7f54b3" stroke-width="2" style="vertical-align:-2px"><circle cx="9" cy="21" r="1"/><circle cx="20" cy="21" r="1"/><path d="M1 1h4l2.68 13.39a2 2 0 0 0 2 1.61h9.72a2 2 0 0 0 2-1.61L23 6H6"/></svg>
                                        <?php echo esc_html__('WooCommerce Live-Events', 'tooltastic-sales-popup'); ?>
                                    </strong>
                                    <small><?php echo esc_html__('Echte Käufe automatisch als Benachrichtigungen senden', 'tooltastic-sales-popup'); ?></small>
                                </span>
                            </label>
                            <p class="tsp-hint tsp-hint--indented">
                                <?php echo esc_html__('Bei jedem WooCommerce-Kauf wird ein Live-Event an Tooltastic gesendet. Deine Besucher sehen dann echte Kaufbenachrichtigungen.', 'tooltastic-sales-popup'); ?>
                                <strong><?php echo esc_html__('Erfordert Pro-Plan.', 'tooltastic-sales-popup'); ?></strong>
                            </p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Save button (visible on all tabs) ──── -->
                <div class="tsp-save">
                    <?php submit_button(esc_html__('Einstellungen speichern', 'tooltastic-sales-popup'), 'primary tsp-btn-save', 'submit', false); ?>
                </div>
            </form>

            <!-- ── Footer ──────────────────────────────── -->
            <div class="tsp-footer">
                <a href="https://tooltastic.eu/de/sales-popup/anleitung/" target="_blank" rel="noopener">
                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M2 3h6a4 4 0 0 1 4 4v14a3 3 0 0 0-3-3H2z"/><path d="M22 3h-6a4 4 0 0 0-4 4v14a3 3 0 0 1 3-3h7z"/></svg>
                    <?php echo esc_html__('Anleitung', 'tooltastic-sales-popup'); ?>
                </a>
                <a href="https://tooltastic.eu/de/sales-popup/faq/" target="_blank" rel="noopener">
                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><path d="M9.09 9a3 3 0 0 1 5.83 1c0 2-3 3-3 3"/><line x1="12" y1="17" x2="12.01" y2="17"/></svg>
                    <?php echo esc_html__('FAQ', 'tooltastic-sales-popup'); ?>
                </a>
                <a href="https://tooltastic.eu" target="_blank" rel="noopener">tooltastic.eu</a>
            </div>
        </div>

        <!-- ── Campaign Card Template (hidden) ───────── -->
        <template id="tsp-campaign-template">
            <?php self::render_campaign_card('__INDEX__', [
                'label'       => '',
                'campaign_id' => '',
                'mode'        => 'all',
                'pages'       => '',
            ]); ?>
        </template>

        <!-- ── Inline JS ─────────────────────────────── -->
        <script>
        (function() {
            'use strict';

            /* ── Tab switching ───────────────────────── */
            var tabs = document.querySelectorAll('.tsp-tab');
            var panels = document.querySelectorAll('.tsp-panel');

            tabs.forEach(function(tab) {
                tab.addEventListener('click', function() {
                    var target = this.getAttribute('data-tab');

                    tabs.forEach(function(t) {
                        t.classList.remove('tsp-tab--active');
                        t.setAttribute('aria-selected', 'false');
                    });
                    this.classList.add('tsp-tab--active');
                    this.setAttribute('aria-selected', 'true');

                    panels.forEach(function(p) {
                        if (p.getAttribute('data-panel') === target) {
                            p.classList.add('tsp-panel--active');
                        } else {
                            p.classList.remove('tsp-panel--active');
                        }
                    });
                });
            });

            /* ── Campaign management ─────────────────── */
            var container = document.getElementById('tsp-campaigns');
            var emptyState = document.getElementById('tsp-empty-state');
            var template = document.getElementById('tsp-campaign-template');
            var addBtn = document.getElementById('tsp-add-campaign');
            var campaignCount = container.querySelectorAll('.tsp-campaign').length;

            function updateIndices() {
                var cards = container.querySelectorAll('.tsp-campaign');
                cards.forEach(function(card, i) {
                    card.setAttribute('data-index', i);
                    card.querySelector('.tsp-campaign-num').textContent = i + 1;

                    // Update name attributes
                    card.querySelectorAll('[name*="tooltastic_sp_campaigns"]').forEach(function(input) {
                        input.name = input.name.replace(/\[\d+\]/, '[' + i + ']');
                    });
                });

                emptyState.style.display = cards.length === 0 ? '' : 'none';
                campaignCount = cards.length;
            }

            function initCard(card) {
                // Collapse/expand
                var header = card.querySelector('.tsp-campaign-header');
                var body = card.querySelector('.tsp-campaign-body');
                var chevron = card.querySelector('.tsp-campaign-chevron');

                header.addEventListener('click', function(e) {
                    // Don't toggle when clicking remove button
                    if (e.target.closest('.tsp-campaign-remove')) return;

                    var isOpen = body.style.display !== 'none';
                    body.style.display = isOpen ? 'none' : '';
                    chevron.classList.toggle('tsp-campaign-chevron--collapsed', isOpen);
                    card.classList.toggle('tsp-campaign--collapsed', isOpen);
                });

                // Remove
                card.querySelector('.tsp-campaign-remove').addEventListener('click', function(e) {
                    e.stopPropagation();
                    card.style.opacity = '0';
                    card.style.transform = 'translateX(20px)';
                    setTimeout(function() {
                        card.remove();
                        updateIndices();
                    }, 200);
                });

                // Mode selector → toggle pages field
                var modeSelect = card.querySelector('.tsp-mode-select');
                var pagesField = card.querySelector('.tsp-pages-field');
                if (modeSelect && pagesField) {
                    function togglePages() {
                        pagesField.style.display = modeSelect.value === 'all' ? 'none' : '';
                    }
                    modeSelect.addEventListener('change', togglePages);
                    togglePages();
                }

                // Label preview in collapsed header
                var labelInput = card.querySelector('.tsp-campaign-label-input');
                var labelPreview = card.querySelector('.tsp-campaign-label-preview');
                if (labelInput && labelPreview) {
                    labelInput.addEventListener('input', function() {
                        labelPreview.textContent = this.value || '<?php echo esc_js(__('Unbenannt', 'tooltastic-sales-popup')); ?>';
                    });
                }
            }

            // Init existing cards
            container.querySelectorAll('.tsp-campaign').forEach(initCard);

            // Add campaign
            addBtn.addEventListener('click', function() {
                var html = template.innerHTML.replace(/__INDEX__/g, campaignCount);
                var wrapper = document.createElement('div');
                wrapper.innerHTML = html.trim();
                var card = wrapper.firstElementChild;

                card.style.opacity = '0';
                card.style.transform = 'translateY(10px)';
                container.appendChild(card);
                initCard(card);
                updateIndices();

                requestAnimationFrame(function() {
                    card.style.transition = 'opacity 0.3s, transform 0.3s';
                    card.style.opacity = '1';
                    card.style.transform = 'translateY(0)';
                });

                // Focus the label input
                var label = card.querySelector('.tsp-campaign-label-input');
                if (label) label.focus();
            });

            /* ── API Key test ────────────────────────── */
            var testBtn = document.getElementById('tsp-test-key');
            var testResult = document.getElementById('tsp-test-result');

            if (testBtn) {
                testBtn.addEventListener('click', function() {
                    var key = document.getElementById('tooltastic_sp_api_key').value.trim();
                    if (!key) {
                        showResult('error', '<?php echo esc_js(__('Bitte API Key eingeben.', 'tooltastic-sales-popup')); ?>');
                        return;
                    }

                    testBtn.disabled = true;
                    testBtn.querySelector('svg').style.animation = 'tsp-spin 1s linear infinite';
                    showResult('loading', '<?php echo esc_js(__('Verbindung wird getestet...', 'tooltastic-sales-popup')); ?>');

                    var data = new FormData();
                    data.append('action', 'tooltastic_sp_test_key');
                    data.append('nonce', '<?php echo esc_js(wp_create_nonce('tooltastic_sp_nonce')); ?>');
                    data.append('api_key', key);

                    fetch(ajaxurl, { method: 'POST', body: data })
                        .then(function(r) { return r.json(); })
                        .then(function(json) {
                            if (json.success) {
                                showResult('success', json.data.message);
                            } else {
                                showResult('error', json.data ? json.data.message : 'Unbekannter Fehler');
                            }
                        })
                        .catch(function() {
                            showResult('error', '<?php echo esc_js(__('Netzwerkfehler.', 'tooltastic-sales-popup')); ?>');
                        })
                        .finally(function() {
                            testBtn.disabled = false;
                            testBtn.querySelector('svg').style.animation = '';
                        });
                });
            }

            function showResult(type, message) {
                testResult.style.display = '';
                testResult.className = 'tsp-message tsp-message--' + type;
                var icons = {
                    success: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>',
                    error: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>',
                    loading: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="animation:tsp-spin 1s linear infinite"><polyline points="23 4 23 10 17 10"/><path d="M20.49 15a9 9 0 1 1-2.12-9.36L23 10"/></svg>',
                };
                testResult.innerHTML = (icons[type] || '') + '<span>' + message + '</span>';
            }
        })();
        </script>
        <?php
    }

    /**
     * Render a single campaign card.
     */
    private static function render_campaign_card($index, $campaign) {
        $label       = $campaign['label'] ?? '';
        $campaign_id = $campaign['campaign_id'] ?? '';
        $mode        = $campaign['mode'] ?? 'all';
        $pages       = $campaign['pages'] ?? '';
        $is_template = ($index === '__INDEX__');
        $num         = $is_template ? '' : ($index + 1);
        ?>
        <div class="tsp-campaign" data-index="<?php echo esc_attr($index); ?>">
            <div class="tsp-campaign-header">
                <span class="tsp-campaign-num"><?php echo esc_html($num); ?></span>
                <span class="tsp-campaign-label-preview"><?php echo esc_html($label ?: __('Unbenannt', 'tooltastic-sales-popup')); ?></span>
                <span class="tsp-campaign-mode-badge">
                    <?php
                    $mode_labels = [
                        'all'     => __('Alle Seiten', 'tooltastic-sales-popup'),
                        'include' => __('Bestimmte Seiten', 'tooltastic-sales-popup'),
                        'exclude' => __('Seiten ausschließen', 'tooltastic-sales-popup'),
                    ];
                    echo esc_html($mode_labels[$mode] ?? $mode_labels['all']);
                    ?>
                </span>
                <span class="tsp-campaign-spacer"></span>
                <span class="tsp-campaign-chevron">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="6 9 12 15 18 9"/></svg>
                </span>
                <button type="button" class="tsp-campaign-remove" title="<?php echo esc_attr__('Kampagne entfernen', 'tooltastic-sales-popup'); ?>">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="3 6 5 6 21 6"/><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/></svg>
                </button>
            </div>

            <div class="tsp-campaign-body">
                <div class="tsp-field-row">
                    <div class="tsp-field tsp-field--half">
                        <label class="tsp-label"><?php echo esc_html__('Bezeichnung', 'tooltastic-sales-popup'); ?></label>
                        <input type="text"
                               name="tooltastic_sp_campaigns[<?php echo esc_attr($index); ?>][label]"
                               value="<?php echo esc_attr($label); ?>"
                               class="tsp-input tsp-campaign-label-input"
                               placeholder="<?php echo esc_attr__('z.B. Shop, Blog, Landing Page', 'tooltastic-sales-popup'); ?>">
                    </div>
                    <div class="tsp-field tsp-field--half">
                        <label class="tsp-label"><?php echo esc_html__('Kampagnen-ID', 'tooltastic-sales-popup'); ?></label>
                        <input type="text"
                               name="tooltastic_sp_campaigns[<?php echo esc_attr($index); ?>][campaign_id]"
                               value="<?php echo esc_attr($campaign_id); ?>"
                               class="tsp-input tsp-input--code"
                               placeholder="xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx">
                    </div>
                </div>

                <div class="tsp-field">
                    <label class="tsp-label"><?php echo esc_html__('Anzeigen auf', 'tooltastic-sales-popup'); ?></label>
                    <select name="tooltastic_sp_campaigns[<?php echo esc_attr($index); ?>][mode]" class="tsp-select tsp-mode-select">
                        <option value="all" <?php selected($mode, 'all'); ?>><?php echo esc_html__('Alle Seiten (Standard)', 'tooltastic-sales-popup'); ?></option>
                        <option value="include" <?php selected($mode, 'include'); ?>><?php echo esc_html__('Nur bestimmte Seiten', 'tooltastic-sales-popup'); ?></option>
                        <option value="exclude" <?php selected($mode, 'exclude'); ?>><?php echo esc_html__('Alle Seiten außer...', 'tooltastic-sales-popup'); ?></option>
                    </select>
                </div>

                <div class="tsp-field tsp-pages-field" <?php echo $mode === 'all' ? 'style="display:none"' : ''; ?>>
                    <label class="tsp-label"><?php echo esc_html__('Seiten (URL-Pfade)', 'tooltastic-sales-popup'); ?></label>
                    <textarea name="tooltastic_sp_campaigns[<?php echo esc_attr($index); ?>][pages]"
                              rows="3"
                              class="tsp-textarea tsp-input--code"
                              placeholder="/shop/*&#10;/produkte/*&#10;/kategorie/schuhe/"><?php echo esc_textarea($pages); ?></textarea>
                    <p class="tsp-hint"><?php echo esc_html__('Ein URL-Pfad pro Zeile. Unterstützt Wildcards: /shop/*', 'tooltastic-sales-popup'); ?></p>
                </div>

                <p class="tsp-hint">
                    <?php
                    printf(
                        esc_html__('Die Kampagnen-ID findest du im %s.', 'tooltastic-sales-popup'),
                        '<a href="https://tooltastic.eu/de/dashboard/sales-popup/" target="_blank" rel="noopener">' . esc_html__('Kampagnen-Manager', 'tooltastic-sales-popup') . '</a>'
                    );
                    ?>
                </p>
            </div>
        </div>
        <?php
    }
}
